import { NextRequest, NextResponse } from "next/server";
import { editAnnonce } from "@/lib/services/annonceService";

type EditableAnnonceFields = {
    titre: string;
    address: string;
    pays: string;
    ville: string;
    codePostal: string;
    description: string;
    prix: number;
    statut: "UNPUBLISHED" | "PUBLISHED";
};

export async function PUT(req: NextRequest, { params }: { params: { id: string } }) {
    try {
        const id = Number(params.id);
        if (!Number.isFinite(id)) {
            return NextResponse.json({ message: "ID invalide." }, { status: 400 });
        }

        const contentType = req.headers.get("content-type") || "";
        if (!contentType.includes("application/json")) {
            return NextResponse.json({ message: "Content-Type attendu: application/json." }, { status: 415 });
        }

        const body = await req.json();

        const data: Partial<EditableAnnonceFields> = {
            titre: typeof body.titre === "string" ? body.titre : undefined,
            address: typeof body.address === "string" ? body.address : undefined,
            pays: typeof body.pays === "string" ? body.pays : undefined,
            ville: typeof body.ville === "string" ? body.ville : undefined,
            codePostal: typeof body.codePostal === "string" ? body.codePostal : undefined,
            description: typeof body.description === "string" ? body.description : undefined,
            prix: Number.isFinite(Number(body.prix)) ? Number(body.prix) : undefined,
            statut: typeof body.statut === "string" ? (body.statut as EditableAnnonceFields["statut"]) : undefined,
        };

        for (const [k, v] of Object.entries(data)) {
            if (v === undefined) {
                return NextResponse.json({ message: `Champ invalide ou manquant: ${k}.` }, { status: 400 });
            }
        }

        const updated = await editAnnonce(id, data);
        return NextResponse.json(updated, { status: 200 });
    } catch (e: any) {
        console.error("Erreur API (PUT /api/annonces/[id]) :", e);
        return NextResponse.json({ message: e?.message ?? "Erreur interne du serveur." }, { status: 500 });
    }
}

export async function PATCH(req: NextRequest, { params }: { params: { id: string } }) {
    try {
        const id = Number(params.id);
        if (!Number.isFinite(id)) {
            return NextResponse.json({ message: "ID invalide." }, { status: 400 });
        }

        const contentType = req.headers.get("content-type") || "";
        if (!contentType.includes("multipart/form-data")) {
            return NextResponse.json({ message: "Content-Type attendu: multipart/form-data." }, { status: 415 });
        }

        const form = await req.formData();
        const file = form.get("imagePrincipale");

        if (!file || !(file instanceof File)) {
            return NextResponse.json(
                { message: "Aucun fichier fourni sous le champ 'imagePrincipale'." },
                { status: 400 }
            );
        }

        const ALLOWED = new Set(["image/jpeg", "image/png", "image/webp"]);
        if (!ALLOWED.has(file.type)) {
            return NextResponse.json({ message: "Type d'image non autorisé." }, { status: 415 });
        }

        const buffer = Buffer.from(await file.arrayBuffer());
        const base64Image = buffer.toString("base64");
        const updated = await editAnnonce(id, { imagePrincipale: base64Image });
        return NextResponse.json(updated, { status: 200 });
        
    } catch (e: any) {
        console.error("Erreur API (PATCH /api/annonces/[id]) :", e);
        return NextResponse.json({ message: e?.message ?? "Erreur interne du serveur." }, { status: 500 });
    }
}
